
FUNCTION RESIDUAL_SMOOTH, prof, kernel, level
;+
; NAME:
;    RESIDUAL_SMOOTH
;
; PURPOSE:
;    To smooth a limb darkening profile so that a wider kernel
;    are used close to disk center.
;
; CALLING SEQUENCE:
;    prof = RESIDUAL_SMOOTH( prof, kernel, level )
;
; INPUTS:
;    PROF:   Limb darkening profile.
;    KERNEL: Kernel size at disk center.
;    LEVEL:  Cutoff level in percent of maximum disk brightness used for
;            the smoothing of the limb darkening function.
;
; OUTPUTS:
;    PROF:   Smoothed limb darkening profile.
;
; KEYWORDS:
;    None.
;
; PROCEDURE:
;    Smoothing kernel size is decreasing proportional to the limb
;    darkening profile out to a level LEVEL percent lower than the
;    disk center level.
;
; MODIFICATION HISTORY:
;    1996-01-01: Anders Johanneson, Big Bear Solar Observatory
;        Original verion of FD_SMOOT.PRO.
;    1999-12-18: Carsten Denker, Big Bear Solar Observatory
;        Intergration in the new BBSO archiving system.
;
;-

    ;----- get range array
cutoff = level / 100.0
k_arr = kernel * ( ( prof / FLOAT( MAX( prof ) ) ) > cutoff - cutoff ) $
    / ( 1.0 - cutoff )

     ;----- smoothing (floating average)
n = N_ELEMENTS( prof )
FOR i = 0, n - 1 DO BEGIN
    kern = BYTE( ROUND( k_arr( i ) ) )
    IF ( kern MOD 2 ) EQ 0 THEN kern = kern + 1
    IF kern GE 3 THEN BEGIN
        sum = 0.0
        FOR k = -kern / 2, kern / 2 DO BEGIN
            j = i + k
            IF j LT 0 THEN j = 0
            IF j GT ( n - 1 ) THEN j = n - 1
            sum = sum + prof( j )
        ENDFOR
        prof( i ) = sum / FLOAT( kern )
    ENDIF
ENDFOR



RETURN, prof

END


