

FUNCTION FLAT_KUHN_LIN_C, flat, bitmap, disp, p, cnt
;+
; NAME:
;    FLAT_KUHN_LIN_C
;
; PURPOSE:
;    Kuhn-Lin style flat field function. See Kuhn, Lin, and Loranz: "Gain
;    Calibrating NonUniform Image-Array Data Using Only the Image Data",
;    Publications of the Astronomy Society of the Pacific, 103: 1097-1108,
;    October 1991.
;
; CATEGORY:
;    BBSO Archiving System.
;    Image Processing.
;
; CALLING SEQUENCE:
;    con = FLAT_KUHN_LIN_C( flat, bitmap, disp, p, cnt )
;
; INPUTS:
;    FLAT:   Array of reduced size input images.
;    BITMAP: Bitmap containing good pixel for each input flat field frame.
;    DISP:   Array of displacements with respect to the centered image, also
;            reduced in size.
;    P:      STRUCTURE image specs.
;
; OUTPUTS:
;    CON:    Constant term of the gain table.
;    CNT:    Valid pixel pair count.

;
; KEYWORDS:
;    None.
;
; MODIFICATION HISTORY:
;    1997-10-27: Anders Johanneson, Big Bear Solar Observatory
;        Original verion of GET_CONST.PRO.
;    1999-12-15: Carsten Denker, Big Bear Solar Observatory
;        Intergration in the new BBSO archiving system, variables are passed
;        as structures.
;
;-

    ;----- definitions
loopct = 0
con    = FLTARR( p.nxx, p.nyy )
dat    = FLTARR( p.nxx, p.nyy, p.n )
cnt    = FLTARR( p.nxx, p.nyy )
ma     = MACHAR( )

    ;----- compute constant term
dat( *, *, 0 ) = ALOG10( flat( *, *, 0 ) > ma.xmin )
FOR iq = 1, p.n - 1 DO BEGIN
    dat( *, *, iq ) = ALOG10( flat( *, *, iq ) > ma.xmin )

    ;----- retrieve good pixel from bitmap
    mskiq = bitmap / ( 2l^iq OR bitmap )
    FOR ir = 0, iq - 1 DO BEGIN
        mskir = bitmap / ( 2l^ir OR bitmap )
        dx = disp( 0, iq ) - disp( 0, ir )
        dy = disp( 1, iq ) - disp( 1, ir )
        jxl = MAX( [ 0, -dx ] )
        jxh = MIN( [ 0, -dx ] ) + p.nxx - 1
        jyl = MAX( [ 0, -dy ] )
        jyh = MIN( [ 0, -dy ] ) + p.nyy - 1
        ixl = MAX( [ 0,  dx ] )
        ixh = MIN( [ 0,  dx ] ) + p.nxx - 1
        iyl = MAX( [ 0,  dy ] )
        iyh = MIN( [ 0,  dy ] ) + p.nyy - 1
        msk = ( mskiq( jxl : jxh, jyl : jyh ) AND mskir( ixl : ixh, iyl : iyh ) )
        diff = ( dat( jxl : jxh, jyl : jyh, iq) - dat( ixl : ixh, iyl : iyh, $
            ir ) ) * msk
        con( jxl : jxh, jyl : jyh ) = con( jxl : jxh, jyl : jyh ) + diff
        con( ixl : ixh, iyl : iyh ) = con( ixl : ixh, iyl : iyh ) - diff
        cnt( jxl : jxh, jyl : jyh ) = cnt( jxl : jxh, jyl : jyh ) + msk
        cnt( ixl : ixh, iyl : iyh ) = cnt( ixl : ixh, iyl : iyh ) + msk
        loopct = loopct + 1
     ENDFOR
ENDFOR



RETURN, con

END
