


FUNCTION FLAT_KUHN_LIN, flat, disp, p
;+
; NAME:
;    FLAT_KUHN_LIN
;
; PURPOSE:
;    Kuhn-Lin style flat field function. See Kuhn, Lin, and Loranz: "Gain
;    Calibrating NonUniform Image-Array Data Using Only the Image Data",
;    Publications of the Astronomy Society of the Pacific, 103: 1097-1108,
;    October 1991.
;
; CATEGORY:
;    BBSO Archiving System.
;    Image Processing.
;
; CALLING SEQUENCE:
;    flat = FLAT_KUHN_LIN( flat, disp, p )
;
; INPUTS:
;    FLAT: Array of reduced size input images.
;    DISP: Array of displacements with respect to the centered image, also
;          reduced in size.
;    P:    STRUCTURE image specs.
;
; OUTPUTS:
;    FLAT: Reduced size flat field frame.
;
; KEYWORDS:
;    None.
;
; MODIFICATION HISTORY:
;    1997-10-27: Anders Johanneson, Big Bear Solar Observatory
;        Original verion of LINFLAT.PRO.
;    1999-12-15: Carsten Denker, Big Bear Solar Observatory
;        Intergration in the new BBSO archiving system, variables are passed
;        as structures, removed the need for the SETBIT.PRO, GETBIT.PRO, and
;        ITERATE.PRO functions.
;
;-

;----- definitions
n = 10    ;----- number of iterations

    ;----- create bitmap by thresholding
rmin = LONG( ROUND ( 0.70 * p.int ) )
rmax = LONG( ROUND ( 1.05 * p.int ) )
bitmap = LONARR ( p.nxx, p.nyy )
IF p.n GE 32 THEN RETURN, -1                 ;----- too many flat field frames
FOR i = 0, p.n - 1 DO bitmap = bitmap + ( ( flat( *, *, i ) GE rmin ) $
    AND ( flat( *, *, i ) LE rmax ) ) * 2l^i
IF p.n LT 16 THEN bitmap = FIX( bitmap )     ;----- INTEGER bitmap
IF p.n LE  8 THEN bitmap = BYTE( bitmap )    ;----- BYTE bitmap

    ;----- get algorithm's constant term and calculate the valid pixel pair count
con = FLAT_KUHN_LIN_C( flat, bitmap, disp, p, cnt )

    ;----- start with unit gain, no need to call DO_ITER now, since the initial
    ;----- guess is gain = 0
gain = con / ( cnt > 1 )

    ;----- iterate the gain
FOR i = 0, n - 1 DO FLAT_KUHN_LIN_I, con, gain, bitmap, cnt, p, disp
flat = 10.^gain * ( bitmap NE 0 )



RETURN, flat
END


