


FUNCTION FLAT_CLEAN, flat, center, p
;+
; NAME:
;    FLAT_CLEAN
;
; PURPOSE:
;    Extrapolates missing values in Kuhn-Lin gain tables. Implicitly assumes
;    that the gain table is defined in the image center.
;
; CATEGORY:
;    BBSO Archiving System.
;    Image Processing.
;
; CALLING SEQUENCE:
;    flat = FLAT_CLEAN( flat, center, p )
;
; INPUTS:
;    FLAT:   Kuhn-Lin gain table with missing values.
;    CENTER: Center coordinates of reduced size image.
;    P:      STRUCTURE image specs.
;
; OUTPUTS:
;    FLAT:   Full size Kuhn-Lin gain table.
;
; KEYWORDS:
;    None.
;
; MODIFICATION HISTORY:
;    1999-12-16: Carsten Denker, Big Bear Solar Observatory
;        Intergration in the new BBSO archiving system, variables are passed
;        as structures, replaces slow CLEANFLAT.PRO
;
;-

;----- definition

fill = flat

;----- distance from image center

d = SHIFT( DIST( p.nxx, p.nyy ), p.nxx / 2, p.nyy / 2 )

;----- sort distance array so that we actually work our way from the inside
;----- to the outside

xy = SORT( d )

;----- replace only missing values

index = WHERE( fill( xy ) EQ 0, n )
xy = xy ( index )

;----- x and y coordinates

xx = xy mod p.nxx
yy = xy / p.nxx

;----- fill the holes (zeros) from the center to the corner

FOR i = 0L, n - 1 DO BEGIN

;----- take care of the different quadrants and avoid problems with image borders

    IF xx( i ) GT p.nxx / 2 THEN x = xx( i ) - 1 ELSE x = xx( i )
    IF yy( i ) GT p.nyy / 2 THEN y = yy( i ) - 1 ELSE y = yy( i )
    fill( xy( i ) ) = MEDIAN ( fill( x : x + 1, y : y + 1 ) )
ENDFOR

;----- smooth image

flat = fill
fill = SMOOTH( fill, 25, /EDGE_TRUNCATE )

;----- use fill image outside limb and blend with sharp flat image

DIST_CIRCLE, cim, [ p.nxx, p.nyy ], center( 0 ), center( 1 )

;;mask = FLOAT( cim LT 0.8 * ( p.nyy / 2 ) )
;;mask = SMOOTH( mask, 25, /EDGE_TRUNCATE )

; use cos^2 profile between 0.7 and 0.9 of solar radius
; instead of sharp mask

wo = WHERE( cim / ( p.nyy / 2. ) GT 0.7 )
mask = FLTARR( p.nxx, p.nyy ) + 1.
mask( wo ) = COS( ( cim( wo ) / ( p.nyy / 2. ) - 0.7 ) / 0.2 * !PI / 2. ) ^ 2.
mask( WHERE( cim / (p.nyy / 2. ) GT 0.9 ) ) = 0.

flat = flat * mask + fill * ( 1.0 - mask )

;----- smooth one final time

;;flat = SMOOTH( flat, 3, /EDGE_TRUNCATE )

;----- resize image to original size

flat = CONGRID( flat, p.nx, p.ny, /INTERP )



RETURN, flat

END




