;----------------------------------------------------------------
; $Id: //depot/idl/IDL_71/idldir/lib/wavelet/source/wv_fn_daubechies.pro#1 $
;
; Copyright (c) 1999-2009, ITT Visual Information Solutions. All
;       rights reserved. Unauthorized reproduction is prohibited.
;+
; NAME:
;    WV_FN_DAUBECHIES
;
; PURPOSE:
;    This function returns the Daubechies wavelet coefficients.
;
; CALLING SEQUENCE:
;    info = WV_FN_DAUBECHIES( Order, Scaling, Wavelet, Ioff, Joff)
;
; INPUTS:
;    Order: This is the order number for the Daubechies wavelet.
;           Order=2 is the "db2" wavelet, and has 2 vanishing moments
;           and 4 coefficients.
;
; OUTPUTS:
;    Scaling: A vector of the scaling (father) coefficients
;
;    Wavelet: A vector of the wavelet (mother) coefficients
;
;    Ioff: The offset index used to center the Scaling support
;
;    Joff: The offset index used to center the Wavelet support
;
; KEYWORD PARAMETERS:
;    None.
;
; RETURN VALUE:
;    Returns a structure with the following information:
;          (this is an example for order=2)
;       info = {family:'Daubechies', $    ; name of wavelet family
;               order_name:'Order', $     ; term used for "order"
;               order_range:[2,24,2], $   ; valid range [first,last,default]
;               order:2, $                ; order number
;               discrete:1, $             ; 0=continuous, 1=discrete
;               orthogonal:1, $           ; 0=nonorthogonal, 1=orthogonal
;               symmetric:0, $            ; 0=asymmetric, 1=symmetric
;               support:3, $              ; support width
;               moments:2, $              ; # of vanishing moments
;               regularity:0.550}         ; # of continuous derivatives
;
; REFERENCE:
;    Daubechies, I., 1992: Ten Lectures on Wavelets, SIAM, p. 195.
;       Daubechies has orders 2-10, although note that Daubechies
;       has multiplied by Sqrt(2).
;
;    Orders 11-24 are from <http://www.isds.duke.edu/~brani/filters.html>
;
; MODIFICATION HISTORY:
;    Written by: Chris Torrence, 1999
;-


;----------------------------------------------------------------
; Daubechies orthogonal asymmetric wavelet coefficients
FUNCTION wv_fn_daubechies_coeff, order
    COMPILE_OPT strictarr, hidden
    CASE (order) OF
    1: coeff = [1d,1d]/SQRT(2d)  ; same as Haar
    2: coeff = [1d + SQRT(3d),3d + SQRT(3d), $
            3d - SQRT(3d),1d - SQRT(3d)]/(4d*SQRT(2d))
    3: BEGIN
        sq10 = SQRT(10d)
        sq5210 = SQRT(5d + 2d*SQRT(10d))
        coeff = [1d + sq10 + sq5210, 5d + sq10 + 3d*sq5210, $
            10d - 2d*sq10 + 2d*sq5210, 10d - 2d*sq10 - 2d*sq5210, $
            5d + sq10 - 3d*sq5210, 1d + sq10 - sq5210]/(16d*SQRT(2d))
        END
    4: coeff = [ $
        0.2303778133088965d,$
        0.7148465705529158d,$
        0.630880767929859d,$
        -0.02798376941686011d,$
        -0.1870348117190932d,$
        0.0308413818355608d,$
        0.03288301166688522d,$
        -0.01059740178506904d]
    5: coeff = [ $
        0.1601023979741924d,$
        0.6038292697971881d,$
        0.7243085284377715d,$
        0.1384281459013217d,$
        -0.2422948870663802d,$
        -0.03224486958463778d,$
        0.07757149384004565d,$
        -0.006241490212798174d,$
        -0.01258075199908194d,$
        0.003335725285473757d]
    6: coeff = [ $
        0.11154074335011d,$
        0.4946238903984554d,$
        0.7511339080210982d,$
        0.315250351709197d,$
        -0.2262646939654429d,$
        -0.1297668675672638d,$
        0.0975016055873231d,$
        0.02752286553030565d,$
        -0.03158203931748625d,$
        0.0005538422011615105d,$
        0.004777257510945544d,$
        -0.001077301085308486d]
    7: coeff = [ $
        0.07785205408500813d,$
        0.3965393194819123d,$
        0.7291320908462274d,$
        0.4697822874051917d,$
        -0.1439060039285563d,$
        -0.2240361849938672d,$
        0.07130921926683042d,$
        0.080612609151082d,$
        -0.03802993693501439d,$
        -0.016574541630667d,$
        0.01255099855609955d,$
        0.0004295779729213739d,$
        -0.001801640704047446d,$
        0.0003537137999745171d]
    8: coeff = [ $
        0.05441584224310704d,$
        0.3128715909143165d,$
        0.6756307362973218d,$
        0.5853546836542239d,$
        -0.01582910525637238d,$
        -0.2840155429615815d,$
        0.0004724845739030209d,$
        0.1287474266204823d,$
        -0.01736930100181088d,$
        -0.04408825393079791d,$
        0.01398102791739956d,$
        0.00874609404740648d,$
        -0.004870352993451852d,$
        -0.000391740373376942d,$
        0.0006754494064506183d,$
        -0.0001174767841247786d]
    9: coeff = [ $
        0.03807794736388813d,$
        0.2438346746126514d,$
        0.6048231236902548d,$
        0.6572880780514298d,$
        0.1331973858249681d,$
        -0.2932737832793372d,$
        -0.0968407832230689d,$
        0.148540749338104d,$
        0.03072568147931585d,$
        -0.06763282906135907d,$
        0.0002509471148277948d,$
        0.02236166212368439d,$
        -0.004723204757752752d,$
        -0.004281503682464633d,$
        0.001847646883056686d,$
        0.0002303857635232296d,$
        -0.0002519631889427889d,$
        0.00003934732031628112d]
    10: coeff = [ $
        0.02667005790054869d,$
        0.188176800077648d,$
        0.527201188931628d,$
        0.6884590394535462d,$
        0.2811723436606982d,$
        -0.2498464243271048d,$
        -0.1959462743773243d,$
        0.127369340335694d,$
        0.0930573646035142d,$
        -0.07139414716638016d,$
        -0.0294575368218849d,$
        0.03321267405931551d,$
        0.003606553566951515d,$
        -0.0107331754833277d,$
        0.001395351747051327d,$
        0.001992405295184184d,$
        -0.0006858566949593225d,$
        -0.0001164668551292262d,$
        0.0000935886703200315d,$
        -0.00001326420289451403d]
    11: coeff = [ $
        0.01869429776144806d,$
        0.1440670211504498d,$
        0.4498997643555165d,$
        0.6856867749154562d,$
        0.4119643689476272d,$
        -0.1622752450269621d,$
        -0.2742308468172826d,$
        0.06604358819685894d,$
        0.1498120124663909d,$
        -0.04647995511648684d,$
        -0.06643878569486228d,$
        0.03133509021904213d,$
        0.02084090436017028d,$
        -0.01536482090617611d,$
        -0.003340858873009247d,$
        0.0049284176560525d,$
        -0.0003085928588149355d,$
        -0.00089302325066525d,$
        0.0002491525235524301d,$
        0.00005443907469928305d,$
        -0.00003463498418694142d,$
        0.000004494274277230458d]
    12: coeff = [ $
        0.01311225795736534d,$
        0.1095662728222715d,$
        0.3773551352176745d,$
        0.657198722584349d,$
        0.5158864784293156d,$
        -0.04476388565908393d,$
        -0.3161784537592869d,$
        -0.02377925725693821d,$
        0.1824786059298069d,$
        0.00535956967427179d,$
        -0.0964321200976865d,$
        0.0108491302560784d,$
        0.04154627749559747d,$
        -0.01221864906995923d,$
        -0.01284082519846823d,$
        0.00671149900888981d,$
        0.002248607241020708d,$
        -0.002179503618657147d,$
        0.000006545128213682533d,$
        0.0003886530628261407d,$
        -0.0000885041092094801d,$
        -0.00002424154575734139d,$
        0.00001277695221955214d,$
        -0.000001529071758089919d]
    13: coeff = [ $
        0.00920213353936357d,$
        0.082861243876398d,$
        0.3119963221728867d,$
        0.6110558511805082d,$
        0.5888895704451372d,$
        0.0869857261666496d,$
        -0.314972907739053d,$
        -0.124576730762086d,$
        0.1794760794355785d,$
        0.07294893365742099d,$
        -0.1058076181950538d,$
        -0.02648840647689916d,$
        0.05613947710301562d,$
        0.002379972253836755d,$
        -0.02383142071161908d,$
        0.003923941449079961d,$
        0.007255589402002825d,$
        -0.002761911234808676d,$
        -0.001315673911943637d,$
        0.000932326130928484d,$
        0.00004925152513188404d,$
        -0.0001651289885636495d,$
        0.00003067853758174376d,$
        0.00001044193057207212d,$
        -0.000004700416479607929d,$
        0.0000005220035098765021d]
    14: coeff = [ $
        0.006461153459818989d,$
        0.0623647588469322d,$
        0.2548502677833766d,$
        0.5543056179241174d,$
        0.6311878490950694d,$
        0.2186706877760189d,$
        -0.2716885522429336d,$
        -0.2180335299738394d,$
        0.138395213856541d,$
        0.1399890165735457d,$
        -0.0867484115685856d,$
        -0.07154895550625034d,$
        0.05523712625188016d,$
        0.02698140830446938d,$
        -0.0301853515397028d,$
        -0.005615049530747707d,$
        0.01278949326524909d,$
        -0.000746218989436958d,$
        -0.003849638867994312d,$
        0.001061691085418039d,$
        0.0007080211541344865d,$
        -0.0003868319473184179d,$
        -0.00004177724577935138d,$
        0.00006875504251988474d,$
        -0.00001033720918460207d,$
        -0.000004389704901652653d,$
        0.000001724994675254821d,$
        -0.000000178713996820958d]
    15: coeff = [ $
        0.004538537356680069d,$
        0.0467433948433292d,$
        0.2060238637760462d,$
        0.4926317712332494d,$
        0.6458131398235114d,$
        0.339002535383428d,$
        -0.19320413905893d,$
        -0.2888825960016258d,$
        0.06528295291444258d,$
        0.1901467139017971d,$
        -0.03966617641454303d,$
        -0.1111209358626346d,$
        0.03387714389352461d,$
        0.05478055052762776d,$
        -0.0257670072911817d,$
        -0.02081005014572826d,$
        0.01508391800773139d,$
        0.005101000354434229d,$
        -0.006487734552531616d,$
        -0.0002417564910950625d,$
        0.001943323977748212d,$
        -0.0003734823537271217d,$
        -0.0003595652439869339d,$
        0.0001558964896924794d,$
        0.00002579269911910246d,$
        -0.00002813329623232866d,$
        0.000003362987176654478d,$
        0.000001811270405641324d,$
        -0.0000006316882317817563d,$
        0.00000006133359905269254d]
    16: coeff = [ $
        0.003189220905181802d,$
        0.0349077141074775d,$
        0.1650642824989111d,$
        0.4303127204089899d,$
        0.6373563289234388d,$
        0.4402902557886062d,$
        -0.0897510867287953d,$
        -0.3270633068118058d,$
        -0.02791820715372535d,$
        0.2111906930487478d,$
        0.02734026408611786d,$
        -0.1323883043443139d,$
        -0.00623972263724492d,$
        0.07592423555847598d,$
        -0.00758897425298305d,$
        -0.03688839741760147d,$
        0.01029765955546528d,$
        0.01399376876290007d,$
        -0.006990014507518413d,$
        -0.003644279596729619d,$
        0.003128023357662664d,$
        0.000407896978913364d,$
        -0.000941021742187743d,$
        0.000114241519113091d,$
        0.0001747872440135933d,$
        -0.00006103596571228747d,$
        -0.00001394566888488284d,$
        0.00001133660857799308d,$
        -0.000001043571333041443d,$
        -0.0000007363656730469882d,$
        0.0000002308784069376313d,$
        -0.00000002109339613774453d]
    17: coeff = [ $
        0.002241806968367765d,$
        0.02598539333038641d,$
        0.1312149014643511d,$
        0.3703507191428474d,$
        0.6109966080619875d,$
        0.5183157592365552d,$
        0.02731497388861195d,$
        -0.3283207398752789d,$
        -0.1265997478695799d,$
        0.1973105883690036d,$
        0.1011354893285621d,$
        -0.1268156885448092d,$
        -0.05709141812622551d,$
        0.081105985705437d,$
        0.02231233608959475d,$
        -0.04692243752178137d,$
        -0.003270955473782776d,$
        0.02273367623263168d,$
        -0.003042989911563062d,$
        -0.00860292137975392d,$
        0.002967996640915282d,$
        0.002301205207197428d,$
        -0.001436845280352317d,$
        -0.0003281325149411173d,$
        0.0004394654201169656d,$
        -0.00002561010931458864d,$
        -0.0000820480308801988d,$
        0.00002318681330990614d,$
        0.000006990600842366534d,$
        -0.000004505942411707292d,$
        0.0000003016549532645506d,$
        0.0000002957700881589635d,$
        -0.0000000842394830828037d,$
        0.000000007267492843919008d]
    18: coeff = [ $
        0.001576310332632241d,$
        0.01928853309434481d,$
        0.1035884729715391d,$
        0.3146789620466176d,$
        0.571826841995251d,$
        0.5718016803655575d,$
        0.147223099399332d,$
        -0.2936540837163994d,$
        -0.2164809618743174d,$
        0.1495339814252923d,$
        0.1670813196471977d,$
        -0.0923318969776604d,$
        -0.1067522571200224d,$
        0.0648872212223416d,$
        0.05705125157931265d,$
        -0.04452614611490133d,$
        -0.02373321210978654d,$
        0.02667070832113655d,$
        0.006262168357742094d,$
        -0.01305148206344844d,$
        0.0001186301071328846d,$
        0.004943344018360076d,$
        -0.001118732786346494d,$
        -0.001340596411265555d,$
        0.0006284657384942994d,$
        0.0002135815764103265d,$
        -0.000198648570821057d,$
        -0.000000153591634265962d,$
        0.00003741238184339052d,$
        -0.00000852060341054129d,$
        -0.00000333263477007513d,$
        0.00000176871313748643d,$
        -0.00000007691633640217469d,$
        -0.0000001176098869880653d,$
        0.00000003068836137122469d,$
        -0.000000002507934683892356d]
    19: coeff = [ $
        0.001108669779715294d,$
        0.01428109865333334d,$
        0.081278114333354d,$
        0.2643884347822977d,$
        0.5244363819574067d,$
        0.6017045501513535d,$
        0.2608949440110274d,$
        -0.2280914100170829d,$
        -0.285838641929714d,$
        0.07465227262054114d,$
        0.2123497512548378d,$
        -0.03351853842979753d,$
        -0.1427856935054576d,$
        0.02758435493215239d,$
        0.0869067594236619d,$
        -0.02650123589611068d,$
        -0.04567422669495623d,$
        0.02162376812192859d,$
        0.01937555029280247d,$
        -0.01398838901012597d,$
        -0.00586692239134182d,$
        0.007040747519198927d,$
        0.0007689543646753964d,$
        -0.002687551858597481d,$
        0.0003418086639330359d,$
        0.0007358025360798398d,$
        -0.0002606761416764582d,$
        -0.0001246007941078683d,$
        0.0000871127066319985d,$
        0.000005105950548947162d,$
        -0.00001664017665533139d,$
        0.000003010964385934741d,$
        0.000001531931507655374d,$
        -0.0000006862755810090276d,$
        0.00000001447088339408005d,$
        0.00000004636937873589416d,$
        -0.000000011164020912898d,$
        0.000000000866684902796269d]
    20: coeff = [ $
        0.0007799530020084384d,$
        0.0105493864101072d,$
        0.06342373157542249d,$
        0.2199419467839922d,$
        0.4726958375631425d,$
        0.6104928215175741d,$
        0.3615021297395791d,$
        -0.139211825416023d,$
        -0.3267863905078842d,$
        -0.01672694530514085d,$
        0.2282909876975237d,$
        0.03985032729018178d,$
        -0.1554585361790331d,$
        -0.02471674917392653d,$
        0.1022916746204368d,$
        0.005632268726873665d,$
        -0.06172283526148656d,$
        0.005874682288534986d,$
        0.03229427583633914d,$
        -0.00878931595226129d,$
        -0.01381051445886118d,$
        0.006721621652169426d,$
        0.004420538864131319d,$
        -0.003581491222634283d,$
        -0.00083156152944895d,$
        0.001392558453825609d,$
        -0.00005349753868856166d,$
        -0.0003851044297986765d,$
        0.0001015328014373285d,$
        0.00006774275277093538d,$
        -0.00003710583043522718d,$
        -0.000004376140493506968d,$
        0.000007241242222701708d,$
        -0.000001011993125412585d,$
        -0.0000006847073928591012d,$
        0.0000002633921999175421d,$
        0.0000000002014328820034285d,$
        -0.0000000181484172957345d,$
        0.000000004056123630675098d,$
        -0.0000000002998833944499773d]
    21: coeff = [ $
        0.0005488240399453808d,$
        0.007776660464348811d,$
        0.04924790475876491d,$
        0.1813601028599902d,$
        0.419688998145241d,$
        0.6015074510688103d,$
        0.4445910837993439d,$
        -0.03572381948901234d,$
        -0.33566575122537d,$
        -0.1123978514710653d,$
        0.2115648260162405d,$
        0.1152333439473735d,$
        -0.1399410472763452d,$
        -0.08177625782428998d,$
        0.09660066710664022d,$
        0.04572352417673011d,$
        -0.06497770623152748d,$
        -0.01865389796875268d,$
        0.03972696757220106d,$
        0.003357765554657301d,$
        -0.02089211624987374d,$
        0.002403482102825579d,$
        0.008988852342563074d,$
        -0.002891344156898007d,$
        -0.002958382842307337d,$
        0.001716612683276365d,$
        0.0006394203289590759d,$
        -0.0006906733219030776d,$
        -0.00003196410553726866d,$
        0.0001936652571660039d,$
        -0.0000363553295677002d,$
        -0.00003499676704742804d,$
        0.00001535487521020741d,$
        2.79033850314008D-6,$
        -3.090027001911602D-6,$
        3.16610662424439D-7,$
        2.99214595113828D-7,$
        -1.000404119487493D-7,$
        -2.254019869522092D-9,$
        7.058055911572644D-9,$
        -1.471958939283684D-9,$
        1.038808947669207D-10]
    22: coeff = [ $
        0.0003862673246197253d,$
        0.005721914066631049d,$
        0.03807032187691932d,$
        0.1483689789282081d,$
        0.3677320057234413d,$
        0.5784372354311235d,$
        0.5079033273631367d,$
        0.07372115020105462d,$
        -0.3127333476121842d,$
        -0.2005720141344328d,$
        0.1640948426591233d,$
        0.179974931810035d,$
        -0.0971123372197599d,$
        -0.1317696149504392d,$
        0.06807740848784511d,$
        0.08455839833964807d,$
        -0.05136497255398131d,$
        -0.04653131832736136d,$
        0.03697137276735332d,$
        0.02058693268949487d,$
        -0.02348031395539096d,$
        -0.006213835918293782d,$
        0.01256489065516637d,$
        0.0003001305020824184d,$
        -0.005455761185358356d,$
        0.001044278408986017d,$
        0.001827032986409597d,$
        -0.000770702101944467d,$
        -0.0004237923063271874d,$
        0.0003286138886837352d,$
        0.0000434593692542139d,$
        -0.00009405347080647135d,$
        0.00001137454223403893d,$
        0.00001737397675279249d,$
        -6.166816318076451D-6,$
        -1.565197277819435D-6,$
        1.295199441207159D-6,$
        -8.78003044824892D-8,$
        -1.283352833826032D-7,$
        3.761280659022215D-8,$
        1.680187679020641D-9,$
        -2.729659356918619D-9,$
        5.33601149622179D-10,$
        -3.60216327759258D-11]
    23: coeff = [ $
        0.0002719278182602901d,$
        0.004203109552950134d,$
        0.02931247643736339d,$
        0.1205254471036576d,$
        0.3184759568589838d,$
        0.5449708209347766d,$
        0.5510501337055957d,$
        0.1813841378320262d,$
        -0.2614398761995617d,$
        -0.2714429864972958d,$
        0.0921245749243952d,$
        0.2235864349031235d,$
        -0.03304774793732929d,$
        -0.164030308293076d,$
        0.02028436820991752d,$
        0.1123069840244809d,$
        -0.0211292480280753d,$
        -0.07021415427385447d,$
        0.02176834655240395d,$
        0.03849895908078205d,$
        -0.01852549112315692d,$
        -0.01753870400805271d,$
        0.01275326613768589d,$
        0.006032371159860696d,$
        -0.00707603267773538d,$
        -0.001134947880346942d,$
        0.003123184807392083d,$
        -0.000246537026777104d,$
        -0.001061334361043996d,$
        0.000319454992361999d,$
        0.0002567865998070605d,$
        -0.0001500373078747796d,$
        -0.00003379188332733358d,$
        0.00004426515179248939d,$
        -2.635561787093299D-6,$
        -8.348692795439998D-6,$
        2.397822036092728D-6,$
        8.148343038224153D-7,$
        -5.339546450998099D-7,$
        1.853340229309923D-8,$
        5.418084825798256D-8,$
        -1.400079829615052D-8,$
        -9.473736128438874D-10,$
        1.050551729533758D-9,$
        -1.93260193304542D-10,$
        1.250331739337031D-11]
    24: coeff = [ $
        0.0001914240079776934d,$
        0.003081894336144903d,$
        0.02248099723913652d,$
        0.09725657409395711d,$
        0.272893661713225d,$
        0.5043448957614517d,$
        0.5749146829767083d,$
        0.2809851510053765d,$
        -0.1872418464658568d,$
        -0.3179111538203686d,$
        0.004781510762825361d,$
        0.2392258659829295d,$
        0.042531243536347d,$
        -0.1711600617797226d,$
        -0.03877318682438014d,$
        0.1210092088290207d,$
        0.02098022912439134d,$
        -0.08215538086453539d,$
        -0.004578395730450242d,$
        0.05129798128535279d,$
        -0.004944235600686442d,$
        -0.02821125709939177d,$
        0.007661004281903052d,$
        0.01304905186620713d,$
        -0.006290964935213451d,$
        -0.004746267936383896d,$
        0.00373576397589871d,$
        0.001153694353296646d,$
        -0.001696334910033699d,$
        -0.00004416435334971148d,$
        0.0005860851561798487d,$
        -0.000118113728929818d,$
        -0.0001459980983446589d,$
        0.00006558881863639525d,$
        0.00002183096348720674d,$
        -0.00002022741617379432d,$
        1.337052417608915D-8,$
        3.900821594914755D-6,$
        -8.979550384702172D-7,$
        -4.032228084773544D-7,$
        2.166180932866001D-7,$
        -5.054643465620961D-10,$
        -2.255577015054618D-8,$
        5.157391468496204D-9,$
        4.748066278754132D-10,$
        -4.024365393060184D-10,$
        6.991284124010881D-11,$
        -4.342457865150871D-12]
    ELSE: coeff = -1
    ENDCASE
    RETURN,coeff
END


;----------------------------------------------------------------
FUNCTION wv_fn_daubechies,order,scaling,wavelet,ioff,joff

    COMPILE_OPT strictarr
    ON_ERROR,2  ; return to caller

;defaults
    order_range = [1,24,2]  ; [first,last,default]
    IF (N_ELEMENTS(order) LT 1) THEN order = order_range[2]  ; default

; check for invalid Order
    order = FIX(order + 1E-5)
    IF ((order LT order_range[0]) OR (order GT order_range[1])) THEN BEGIN
        MESSAGE,/INFO,'Order out of range, reverting to default...'
        order = order_range[2]  ; default
    ENDIF

; Regularity = # of continuous derivatives
; orders  0-4 from Daubechies (1992) p. 239
; orders 5-24 from Taswell (1998) p. 36 "DROMA"
    regularity24 = [0d,0d,0.550d,1.088d, $ ; order 0-3
        1.618d,1.969d,2.189d,2.460d, $   ; 4-7
        2.761d,3.074d,3.381d,3.603d, $   ; 8-11
        3.833d,4.073d,4.317d,4.558d, $   ; 12-15
        4.791d,5.014d,5.239d,5.465d, $   ; 16-19
        5.691d,5.916d,6.138d,6.360d, $   ; 20-23
        6.581d] ; 24
    IF (order LE 24) THEN BEGIN
        regularity = regularity24[order]
    ENDIF ELSE BEGIN ; orders > 24 from Daubechies (1992) p. 226
        regularity = (0.2*order)
        regularity = regularity > regularity24[24] ; CT assumption
    ENDELSE

; construct Info structure
    info = {family:'Daubechies', $
        order_name:'Order', $
        order_range:order_range, $
        order:order, $
        discrete:1, $
        orthogonal:1, $
        symmetric:0, $
        support:FIX(order*2-1), $
        moments:FIX(order), $
        regularity:regularity}

    IF (N_PARAMS() LT 1) THEN RETURN, info ; don't bother with rest

; choose scaling coefficients
    scaling = WV_FN_DAUBECHIES_COEFF(order)

; construct wavelet coefficients & offsets
    n = N_ELEMENTS(scaling)
    wavelet = REVERSE(scaling)*((-1)^LINDGEN(n))
    ioff = -n/2 + 2  ; offset for scaling
    joff = ioff      ; offset for wavelet

    RETURN, info
END


