;  $Id: //depot/idl/IDL_71/idldir/examples/doc/widgets/context_text_example.pro#1 $

;  Copyright (c) 2005-2009, ITT Visual Information Solutions. All
;       rights reserved.
; 
; This file provides an example of the use of the context-menu widget
; within a text widget. Context menus are discussed in detail in
; the "Widget Application Techniques" chapter of _Building IDL Applications._
;
; To see the context menu in action, run this program and click the
; right mouse button in the text field that is displayed.
;
; The example uses a set of event-handler routines to print information
; to IDL's output when items are selected from the context menu. In addition,
; selecting "row" or "column" from the context menu causes a change in the
; widget GUI itself. In a more realistic situation, the event handler routines
; would perform more sophisticated actions.

; Event handler routine for the "Column" button in the context menu.
PRO CTE_ColumnEvent, event

  COMPILE_OPT hidden
  
  ; Obtain the location variable from the UVALUE of the text widget.
  locationText = WIDGET_INFO(event.TOP, FIND_BY_UNAME = 'xyText')
  WIDGET_CONTROL, locationText, GET_UVALUE = location

  ; If location index is set to "Row", change it to "Column".
  IF (location[2] EQ 1) THEN BEGIN
    titleLabel = WIDGET_INFO(event.TOP, FIND_BY_UNAME = 'xyLabel')
    WIDGET_CONTROL, titleLabel, SET_VALUE = 'Column:  '
    location[2] = 0
  ENDIF ELSE RETURN

  ; Store updated location variable in the UVALUE of the text widget.
  WIDGET_CONTROL, locationText, SET_UVALUE = location

END

; Event handler routine for the "Row" button in the context menu.
PRO CTE_RowEvent, event

  COMPILE_OPT hidden

  ; Obtain the location variable from the UVALUE of the text widget.
  locationText = WIDGET_INFO(event.TOP, FIND_BY_UNAME = 'xyText')
  WIDGET_CONTROL, locationText, GET_UVALUE = location

  ; If location index is set to "Column" change it to "Row".
  IF (location[2] EQ 0) THEN BEGIN
    titleLabel = WIDGET_INFO(event.TOP, FIND_BY_UNAME = 'xyLabel')
    WIDGET_CONTROL, titleLabel, SET_VALUE = 'Row:  '
    location[2] = 1
  ENDIF ELSE RETURN

  ; Store updated location variable in the UVALUE of the text widget.
  WIDGET_CONTROL, locationText, SET_UVALUE = location

END

; Event handler routine for the "Done" button in the context menu.
PRO CTE_DoneEvent, event

  COMPILE_OPT hidden

  ; Destroy the top level base.
  WIDGET_CONTROL, event.TOP, /DESTROY

END

; Event handler routine for the events generated by the text widget.
; This routine is called when the user either clicks the right mouse
; button or edits the text in the text widget.
PRO CTE_TextEvents, event

  COMPILE_OPT hidden

  ; If the event was a right-click, display the context menu.
  IF (TAG_NAMES(event, /STRUCTURE_NAME) EQ 'WIDGET_CONTEXT') THEN $
    BEGIN

    ; Obtain the widget ID of the context menu base.
    contextBase = WIDGET_INFO(event.TOP, FIND_BY_UNAME = 'contextMenu')

    ; Display the context menu and send its events to
    ; the other event handler routines.
    WIDGET_DISPLAYCONTEXTMENU, event.ID, event.X, event.Y, contextBase
  ENDIF

  ; If the text was edited, get the new text from the text widget.
  WIDGET_CONTROL, event.ID, GET_VALUE = textString

  ; If the user enters a string rather than a number, an error occurs.
  ; Check for IO errors and jump directly to warning dialog.
  ON_IOERROR, badnum

  ; Check to see if the input string was within the allowed range.
  IF ((FIX(textString) GE 0) AND (FIX(textString) LE 360)) THEN BEGIN
    textValue = FIX(textString)
  ENDIF ELSE BEGIN
  badnum:
    ; If the user enters a value outside the allowed range, show a
    ; dialog and set the value equal to zero.
    dialog=DIALOG_MESSAGE('Please enter a number between 0 and 360')
    textValue = 0
    WIDGET_CONTROL, event.ID, SET_VALUE=STRTRIM(STRING(textValue), 1)
  ENDELSE

  ; Display the value.
  PRINT, ' '
  PRINT, 'Text Value = ', textValue

  ; Obtain the location variable from the UVALUE of the text widget.
  WIDGET_CONTROL, event.ID, GET_UVALUE = location

  ; Determine if inputed value should be column or row.
  IF(location[2] EQ 0) THEN location[0] = textValue $
    ELSE location[1] = textValue

  ; Output resulting location.
  PRINT, '    Column = ', location[0]
  PRINT, '       Row = ', location[1]

  ; Store updated location variable in the UVALUE of the
  ; text widget.
  WIDGET_CONTROL, event.ID, SET_UVALUE = location

END

; Main Routine: create the GUI.
PRO context_text_example

  ; Create the top level (background) base.
  topLevelBase = WIDGET_BASE(/COLUMN)

  ; Initialize location variable.  This variable contains the column value,
  ; the row value, and a location index. The location index determines if the
  ; text value represents a column value, or it represents a row value.
  column = 180
  row = 180
  locationIndex = 0
  location = [column, row, locationIndex]

  ; Set initial title of the label for the text widget.
  title = 'Column:  '

  ; Create a base to contain an informational message.
  bigBase = WIDGET_BASE(topLevelBase, /COLUMN, /FRAME)

  ; Create an informational message label.
  bigLabel = WIDGET_LABEL(bigBase, VALUE = 'Enter a number between 1 - 360', $
    /DYNAMIC_RESIZE)

  ; Create a base to contain the text widget and its label.
  textBase = WIDGET_BASE(bigBase, /ROW)

  ; Initialize the label of the text widget.
  titleLabel = WIDGET_LABEL(textBase, VALUE = title, $
    /DYNAMIC_RESIZE, UNAME = 'xyLabel')

  ; Initialize the text widget.
  locationText = WIDGET_TEXT(textBase, VALUE = STRTRIM(column, 2), $
    /EDITABLE, UNAME = 'xyText', /CONTEXT_EVENTS, $
    UVALUE = location, EVENT_PRO = 'CTE_TextEvents')

  ; Initialize the base for the context menu.
  contextBase = WIDGET_BASE(topLevelBase, /CONTEXT_MENU, UNAME = 'contextMenu')

  ; Initialize the buttons of the context menu.
  columnButton = WIDGET_BUTTON(contextBase, $
    VALUE = 'Column', EVENT_PRO = 'CTE_ColumnEvent')
  rowButton = WIDGET_BUTTON(contextBase, $
    VALUE = 'Row', EVENT_PRO = 'CTE_RowEvent')
  doneButton = WIDGET_BUTTON(contextBase, VALUE = 'Done', $
    /SEPARATOR, EVENT_PRO = 'CTE_DoneEvent')

  ; Display the GUI.
  WIDGET_CONTROL, topLevelBase, /REALIZE

  ; Handle the events from the GUI.
  XMANAGER, 'context_text_example', topLevelBase

END

