;  $Id: //depot/idl/IDL_71/idldir/examples/doc/widgets/context_draw_example.pro#1 $

;  Copyright (c) 2005-2009, ITT Visual Information Solutions. All
;       rights reserved.
; 
; This file provides an example of the use of the context-menu widget
; within a draw widget. Context menus are discussed in detail in
; the "Widget Application Techniques" chapter of _Building IDL Applications._
;
; To see the context menu in action, run this program and click the
; right mouse button in the draw widget.

; Event handler routine for the "XLOADCT" button on the context menu.
PRO CDE_LoadCTEvent, event

  COMPILE_OPT hidden

  ; Display the XLOADCT utility to allow the user to
  ; change the current color table.
  XLOADCT, /BLOCK, GROUP = event.ID

  ; Obtain the window ID of the draw widget.
  imageDraw = WIDGET_INFO(event.TOP, FIND_BY_UNAME = 'imageDisplay')
  WIDGET_CONTROL, imageDraw, GET_VALUE = windowDraw

  ; Obtain the image to redisplay it with the updated
  ; color table from XLOADCT utility.
  WIDGET_CONTROL, event.TOP, GET_UVALUE = image

  ; Redisplay the image with the updated color table.
  WSET, windowDraw
  TV, image

END

; Event handler routine for the "XPALETTE" button on the context menu.
PRO CDE_PaletteEvent, event

  COMPILE_OPT hidden

  ; Display the XPALETTE utility to allow the user to
  ; modify some or all of the current color table.
  XPALETTE, /BLOCK, GROUP = event.ID

  ; Obtain the window ID of the draw widget.
  imageDraw = WIDGET_INFO(event.TOP, FIND_BY_UNAME = 'imageDisplay')
  WIDGET_CONTROL, imageDraw, GET_VALUE = windowDraw

  ; Obtain the image to redisplay it with the updated
  ; color table from XPALETTE utility.
  WIDGET_CONTROL, event.TOP, GET_UVALUE = image

  ; Redisplay the image with the updated color table.
  WSET, windowDraw
  TV, image

END

; Event handler routine for the "Done" button on the context menu.
PRO CDE_DoneEvent, event

  COMPILE_OPT hidden

  ; Destroy the top level base.
  WIDGET_CONTROL, event.TOP, /DESTROY

END

; Event handler routine for events generated by the draw widget.
PRO CDE_DrawEvents, event

  COMPILE_OPT hidden

  ; If either a left- or right-click occurs, obtain the image to
  ; determine the value of the pixel at the location under the
  ; cursor.
  WIDGET_CONTROL, event.TOP, GET_UVALUE = image

  ; Print the location of the cursor and the value of the pixel
  ; at that location. Note that we pad the byte value of the pixel
  ; with spaces to get the digits to line up properly.
  PRINT, ' '
  PRINT, 'Column:', event.X
  PRINT, '   Row:', event.Y
  PRINT, ' Value:        ', image[event.X, event.Y]

  ; If the event was a right-click, display the context menu. Note
  ; that because draw widgets cannot generate context events, we must
  ; instead configure the draw widget to generate button events and
  ; watch for a right-button release event in order to display the
  ; context menu. See _Building IDL Applications_ for additional
  ; details on this technique.
  IF (event.RELEASE EQ 4) THEN BEGIN
    ; Obtain the widget ID of the context menu base.
    contextBase = WIDGET_INFO(event.TOP, FIND_BY_UNAME = 'drawContext')
    ; Display the context menu.
    WIDGET_DISPLAYCONTEXTMENU, event.ID, event.X, event.Y, contextBase
  ENDIF

END

; Main Routine: create the GUI.
PRO context_draw_example

  ; Retrieve the path to the file containing the image to be displayed.
  file = FILEPATH('worldelv.dat', SUBDIRECTORY = ['examples', 'data'])

  ; Initialize the image size parameter.
  imageSize = [360, 360]

  ; Import the image from the file.
  image = READ_BINARY(file, DATA_DIMS = imageSize)

  ; Create the top level (background) base.
  topLevelBase = WIDGET_BASE(/COLUMN)

  ; Create a draw widget to display the image, enabling button events.
  imageDraw = WIDGET_DRAW(topLevelBase, /BUTTON_EVENTS, XSIZE = imageSize[0], $
    YSIZE = imageSize[1], EVENT_PRO = 'CDE_DrawEvents', UNAME = 'imageDisplay')

  ; Create the base for the context menu.
  contextBase = WIDGET_BASE(topLevelBase, /CONTEXT_MENU, UNAME = 'drawContext')

  ; Create the buttons for the context menu.
  loadCTButton = WIDGET_BUTTON(contextBase, VALUE = 'XLOADCT', $
    EVENT_PRO = 'CDE_LoadCTEvent')
  paletteButton = WIDGET_BUTTON(contextBase, VALUE = 'XPALETTE', $
    EVENT_PRO = 'CDE_PaletteEvent')
  doneButton = WIDGET_BUTTON(contextBase, VALUE = 'Done', $
    /SEPARATOR, EVENT_PRO = 'CDE_DoneEvent')

  ; Display the GUI.
  WIDGET_CONTROL, topLevelBase, /REALIZE

  ; Set the UVALUE of the top level base to the variable containing the
  ; image, so it can be accessed within the event handler routines.
  WIDGET_CONTROL, topLevelBase, SET_UVALUE = image

  ; Obtain the window ID of the draw widget.
  WIDGET_CONTROL, imageDraw, GET_VALUE = windowDraw

  ; Set the display to the window within the draw widget.
  WSET, windowDraw

  ; Initialize the display.
  DEVICE, DECOMPOSED = 0
  LOADCT, 5

  ; Display the image in the window of the draw widget.
  TV, image

  ; Determine the center location of the image display.
  column = imageSize[0]/2
  row = imageSize[1]/2

  ; Initially show the cursor in the center of the image display.
  TVCRS, column, row

  ; Handle the events from the GUI.
  XMANAGER, 'context_draw_example', topLevelBase, /NO_BLOCK

END

